// charon_run
//
// Copyright (c) 2015, Peter Sewell, Kayvan Memarian, David Chisnall
// All rights reserved. Released under the BSD 3-Clause License in LICENSE.


// as an exercise: code in a mostly-pure FP style in C
// (assume running all tests will not exhaust memory, so no need for free)

#define _GNU_SOURCE  //necessary for asprintf in gcc

#include <assert.h>
#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

// for linux
#include <sys/resource.h>
#include <sys/wait.h>

#include "getopts_.h"
#include "types.h"
#include "jsmn.h"


typedef enum execution_mode_e {
  COMPILE_ONLY,
  RUN_ONLY,
  COMPILE_AND_RUN
} execution_mode_t;



/* ***************************************************************** */
/*  error handling                                                   */
/* ***************************************************************** */

void perror_exit(const char *s) {
  perror(s);
  exit(EXIT_FAILURE);
}

void perror2_exit(const char *s, char *s2) {
  fprintf(stderr,"File %s: ",s2);
  perror(s);
  exit(EXIT_FAILURE);
}

void myerror_exit(const char *s) {
  fprintf(stderr,"%s",s);
  exit(EXIT_FAILURE);
}


/* ***************************************************************** */
/*  string manipulation                                              */
/* ***************************************************************** */

string_t app(string_t s1, string_t s2) {
  size_t len;
  len = s1->data_length + s2->data_length;
  char *buf;
  buf = malloc(len);
  string_t s;
  s = malloc(sizeof(struct string_s)); 
  memcpy(buf,s1->buffer,s1->data_length);
  memcpy(buf+s1->data_length,s2->buffer,s2->data_length);
  s->buffer = buf;
  s->buffer_length = len;
  s->data_length = len; 
  return(s);  
}

// construct a new string by appending a C constant string (not including its implicit 0 terminator)
string_t string_const_append(string_t s1, char *cs) {
  size_t len,len2;
  len2=strlen(cs);
  len = s1->data_length + len2;
  char *buf;
  buf = malloc(len);
  string_t s;
  s = malloc(sizeof(struct string_s)); 
  memcpy(buf,s1->buffer,s1->data_length);
  memcpy(buf+s1->data_length,cs,len2);
  s->buffer = buf;
  s->buffer_length = len;
  s->data_length = len; 
  return(s);  
}

// construct a new string by appending a C constant string (including its implicit 0 terminator)
string_t string_const_append_term(string_t s1, char *cs) {
  size_t len,len2;
  len2=strlen(cs)+1;
  len = s1->data_length + len2;
  char *buf;
  buf = malloc(len);
  string_t s;
  s = malloc(sizeof(struct string_s)); 
  memcpy(buf,s1->buffer,s1->data_length);
  memcpy(buf+s1->data_length,cs,len2);
  s->buffer = buf;
  s->buffer_length = len;
  s->data_length = len; 
  return(s);  
}

// construct a new string by removing a 0 terminator)
void pp_debug_string(string_t s);

string_t string_strip_term(string_t s1) {
  assert(s1->data_length >= 1);
  //if (*(s1->buffer + s1->data_length -1) != 0) {
  //  printf("\n**************\n"); pp_debug_string(s1); }
  assert(*(s1->buffer + s1->data_length - 1) == 0);
  int len;
  len = s1->data_length - 1;
  char *buf;
  buf = malloc(len);
  string_t s;
  s = malloc(sizeof(struct string_s)); 
  memcpy(buf,s1->buffer,len);
  s->buffer = buf;
  s->buffer_length = len;
  s->data_length = len; 
  return(s);  
}


string_t string_create(int len) {
  char *buf;
  buf = malloc(len);
  string_t s;
  s = malloc(sizeof(struct string_s)); 
  s->buffer = buf;
  s->buffer_length = len;
  s->data_length = 0; 
  return(s);  
}

// *cs should be null-terminated, but the result does not contain that null
string_t string_from_constant(char *cs) {
  size_t len;
  len = strlen(cs);
  char *buf;
  buf = malloc(len);
  string_t s;
  s = malloc(sizeof(struct string_s)); 
  memcpy(buf,cs,len);
  s->buffer = buf;
  s->buffer_length = len;
  s->data_length = len; 
  return(s);  
}

string_t string_option_from_constant(char *cs) {
  if (cs == NULL) 
    return(NULL);
  else
    return(string_from_constant(cs));
}

string_t string_from_memory(char *cstart, char *cend) {
  size_t len;
  len = cend-cstart;
  char *buf;
  buf = malloc(len + 1);
  string_t s;
  s = malloc(sizeof(struct string_s)); 
  memcpy(buf,cstart,len);
  s->buffer = buf;
  s->buffer[len] = '\0'; /* for printf() */
  s->buffer_length = len;
  s->data_length = len; 
  return(s);  
}


/* DEAD
string_t std_string_from_file(char *filename) {
  FILE *fp;
  
  // open file for reading
  if ( !(fp = fopen(filename, "r")) )
    perror2_exit("fopen() failed\n", filename);
  
  // find size of file
  
} */


string_t string_from_file(char *cs) {
  int fd;
  
//  printf("%s\n", cs);
//  fflush(stdout);
  
  // open file for reading
  fd = open(cs,O_RDONLY);
  if (fd == -1) perror2_exit("open returned -1\n",cs);

  // find size of file
  struct stat stat_buf;
  int res;
  res = fstat(fd, &stat_buf);
  if (res == -1) perror2_exit("fstat returned -1",cs);
  off_t len;
  len = stat_buf.st_size;

  // allocate buffer
  char *data = (char*)malloc(len);

  // read data
  ssize_t count = read(fd, data, len);
  if (count == -1) perror2_exit("read returned -1",cs);
  if (count != len) perror2_exit("read read too few bytes",cs);

  // allocate string and populate
  string_t s = (struct string_s *)malloc(sizeof(string_st));
  s->buffer = data;
  s->buffer_length = len;
  s->data_length = len;

  // close file
  fd = close(fd);
  if (fd == -1) perror2_exit("close returned -1",cs);

  return(s);
}

bool string_prefix(string_t s1, string_t s2) {
  return(0==memcmp(s1->buffer, s2->buffer, s1->data_length));
}

// Print a possibly not null-terminate string_t
#define PRINTF_STRING_T_ARG(s) (s)->data_length, (s)->buffer
void fprintf_string(FILE* f, string_t s) {
  fprintf(f, "%.*s", PRINTF_STRING_T_ARG(s));
}

/* ***************************************************************** */
/*  pp code                                                          */
/* ***************************************************************** */

void pp_debug_string(string_t s) {
  printf("RAW: buffer=%p buffer_length=%i data_length=%i data=**",(void*)s->buffer,s->buffer_length,s->data_length);
  int i;
  for (i=0; i<s->data_length; i++) {
    char c;
    c = s->buffer[i];
    if (c>=' ' && c <127) 
      printf("%c",s->buffer[i]);
    else
      printf("[%02x]",s->buffer[i]);
  }
  printf("**");
  fflush(stdout);
}

void pp_char(FILE *stream, unsigned char c) {
  if (c=='\\') 
    fprintf(stream,"\\\\");
  else if (c=='\"')
    fprintf(stream,"\\\"");
  else if (c=='\n')
    fprintf(stream,"\\n");
  else if (c=='\t')
    fprintf(stream,"\\t");
  else if (c>=0x20 || c>=0x7f) 
    fprintf(stream,"%c",c);
  else 
    fprintf(stream,"\\u%04x",(int)c);
}


// TODO escape...
void pp_string(FILE *stream, string_t s) {
  int i;
  //pp_debug_string(s);
  fprintf(stream,"\"");
  for (i=0; i<s->data_length; i++)
    pp_char(stream,(unsigned char)s->buffer[i]);
  fprintf(stream,"\"");
  fflush(stdout);
}

void pp_raw_string(FILE *stream, string_t s) {
  int i;
  //pp_debug_string(s);
  for (i=0; i<s->data_length; i++)
    fprintf(stream,"%c",(unsigned char)s->buffer[i]);
  fflush(stdout);
}


void pp_indent(FILE*stream, int indent) {
  int i;
  for (i=0;i<indent;i++) 
    fprintf(stream," ");
}

void pp_key(FILE*stream, int indent, char *key) {
  pp_indent(stream,indent);
  fprintf(stream,"\"%s\": ",key);
  //  fprintf(stream,"%s: ",key);
  fflush(stdout);
}

void pp_keystring(FILE*stream, int indent, string_t keystring) {
  pp_indent(stream,indent);
  //pp_string(stream,keystring);
  fprintf(stream,"\"");
  //pp_string(stream,keystring);
  fprintf(stream,"\": ");
  fflush(stdout);
}

void pp_itemend(FILE*stream, bool last) {
  if (last) 
    fprintf(stream,"\n");
  else 
    fprintf(stream,",\n");
}

void pp_end_with_optional_newline(FILE*stream, int indent, bool last, bool lastnewline) {
  pp_indent(stream,indent);
  if (last) 
    fprintf(stream,"}");
  else 
    fprintf(stream,"},");
  if (lastnewline) 
    fprintf(stream,"\n");
}
void pp_end(FILE*stream, int indent, bool last) {
  pp_end_with_optional_newline(stream, indent, last, true);
}

void pp_start(FILE*stream) {
  fprintf(stream, "{\n");
}

void pp_key_stringvalue(FILE *stream, int indent, char *key, string_t s, bool last) {
  pp_key(stream,indent,key);
  pp_string(stream,s);
  pp_itemend(stream,last);
} 

void pp_key_stringoptionvalue(FILE *stream, int indent, char *key, string_t s, bool last) {
  pp_key(stream,indent,key);
  if (s!=NULL) 
    pp_string(stream,s);
  else
    fprintf(stream,"\"\"");
  pp_itemend(stream,last);
} 


void pp_keystring_stringvalue(FILE *stream, int indent, string_t keystring, string_t s, bool last) {
  pp_keystring(stream,indent,keystring);
  pp_string(stream,s);
  pp_itemend(stream,last);
} 


void pp_key_bigstringvalue(FILE *stream, int indent, char *key, string_t s, bool last) {
  pp_key(stream,indent,key); fprintf(stream,"\n");
  pp_string(stream,s);
  pp_itemend(stream,last);
} 

void pp_key_intvalue(FILE *stream, int indent, char *key, int i, bool last) {
  pp_key(stream,indent,key);
  fprintf(stream,"\"%i\"",i);
  pp_itemend(stream,last);
} 

void pp_key_timevalue(FILE *stream, int indent, char *key, struct timeval tv, bool last) {
  pp_key(stream,indent,key);
  fprintf(stream,"\"%lu.%06lu\"",tv.tv_sec,(long)tv.tv_usec);
  pp_itemend(stream,last);
} 

void pp_sources(FILE *stream, int indent, sources_t s, bool last) {
  pp_start(stream);
  pp_key_stringvalue(stream,indent+2,"src_dir",s->src_dir,false);
  pp_key_stringvalue(stream,indent+2,"src_main",s->src_main,true);
#ifdef NOTYET
  pp_key_stringvalue(stream,indent+2,"src_main",s->src_main,s->src_aux == NULL);
  string_list_t src_aux = s->src_aux;
  while (src_aux) {
    pp_key_stringvalue(stream,indent+2,"src_aux",src_aux->string,src_aux->next == NULL);
    src_aux = src_aux->next;
    /* TODO: this may cause repeated keys once there are tests with more than 2 srcs */
    assert(!src_aux && "Multiple src_aux not handled yet");
  }
#endif
  pp_end(stream,indent,last);
}

void pp_tool(FILE *stream, int indent, tool_t t, bool last) {
  pp_start(stream);
  pp_key_stringvalue(stream,indent+2,"tool_name",t->tool_name,false);
  pp_key_stringvalue(stream,indent+2,"tool_args",t->tool_args,false);
  pp_key_stringvalue(stream,indent+2,"tool_instance_name",t->tool_instance_name,false);
  if (t->tool_single_phase) 
    pp_key_stringvalue(stream,indent+2,"tool_single_phase",string_from_constant("true"),false);
  else
    pp_key_stringvalue(stream,indent+2,"tool_single_phase",string_from_constant("false"),false);
  if (t->tool_run_prefix)
    pp_key_stringvalue(stream,indent+2,"tool_run_prefix",t->tool_run_prefix,true);
  else
    pp_key_stringvalue(stream,indent+2,"tool_run_prefix",string_from_constant(""),true);
  pp_end(stream,indent,last);
}

void pp_recipe(FILE *stream, int indent, recipe_t r, bool last) {
  pp_start(stream);
  pp_key_stringvalue(stream, indent+2, "test_instance_name", r->test_instance_name,false);
  pp_key(stream,indent+2,"sources"); pp_sources(stream,indent+2,r->sources,false); 
  pp_key(stream,indent+2,"tool"); pp_tool(stream,indent+2,r->tool,true); 
  pp_end(stream,indent,last);
}


void pp_host(FILE *stream, int indent, host_t h, bool last) {
  pp_start(stream);
  pp_key_stringvalue(stream,indent+2,"host_name",h->host_name,false);
  pp_key_stringvalue(stream,indent+2,"host_os",h->host_os,false);
  pp_key_stringvalue(stream,indent+2,"host_os_version",h->host_os_version,false);
  pp_key_stringvalue(stream,indent+2,"host_hardware_platform",h->host_hardware_platform,false);
  pp_key_stringvalue(stream,indent+2,"host_cpuinfo",h->host_cpuinfo,true);
  pp_end(stream,indent,last);
}

void pp_host_option(FILE *stream, int indent, host_t h, bool last) {
  if (h==NULL) {
    fprintf(stream,"\"NULL\",\n");
  }
  else
    pp_host(stream,indent,h,last);
}

void pp_output(FILE *stream, int indent, output_t o, bool last, bool lastnewline) {
  pp_start(stream);
  pp_key_stringvalue(stream,indent+2,"command",o->command,false);
  pp_key_stringvalue(stream,indent+2,"stdout",o->command_stdout,false);
  pp_key_stringvalue(stream,indent+2,"stderr",o->command_stderr,false);
  pp_key_intvalue(stream,indent+2,"exit_code",o->exit_code,false);
  pp_key_stringoptionvalue(stream,indent+2,"signals",o->signals,false);
  pp_key_stringvalue(stream,indent+2,"start_time",o->start_time,false);
  pp_key_timevalue(stream,indent+2,"duration",o->duration,true);
  pp_end_with_optional_newline(stream,indent,last,lastnewline);
}

void pp_output_option(FILE *stream, int indent, output_t o, bool last) {
  if (o==NULL) {
    fprintf(stream,"\"NULL\",\n");
  }
  else
    pp_output(stream,indent,o,last,true);
}

void pp_observation(FILE *stream, observation_t o, bool last) {
  pp_start(stream);
  pp_key(stream,2,"test_recipe"); pp_recipe(stream,2,o->test_recipe,false); 
  pp_key_stringvalue(stream, 2, "test_name", o->test_name,false);
  pp_key_stringvalue(stream, 2, "tool_version", o->tool_version,false);
  pp_key(stream,2,"compile_host"); pp_host_option(stream,2,o->compile_host,false); 
  pp_key(stream,2,"compile_output"); pp_output_option(stream,2,o->compile_output,false); 
  pp_key_stringoptionvalue(stream, 2, "binary_filename", o->binary_filename,false);
  pp_key(stream,2,"execute_host"); pp_host_option(stream,2,o->execute_host,false); 
  pp_key(stream,2,"execute_output"); pp_output_option(stream,2,o->execute_output,false); 
  pp_key_bigstringvalue(stream, 2, "source_hashes", o->source_hashes,false);
  pp_key_bigstringvalue(stream, 2, "sources", o->sources,true);
  // TODO: assembly?
  // TODO: binary?
  pp_end(stream,0,last);
} 

void pp_observation_from_intermediate(FILE *stream, host_t h, int_t i, output_t o, bool last) {
  pp_raw_string(stream,i->int_first_fields);
  //fprintf(stream,"X");
  pp_key(stream,0,"execute_host"); pp_host(stream,2,h,false);
  pp_key(stream,2,"execute_output"); pp_output(stream,2,o,true,false); 
  //fprintf(stream,"Y");
  pp_raw_string(stream,i->int_last_fields);
  if (!last) 
    fprintf(stream,",");
  fprintf(stream,"\n");
}
  

void pp_int(FILE *stream, int_t i, bool last) {
  pp_start(stream);
  pp_key_stringvalue(stream, 2, "int_test_instance_name", i->int_test_instance_name,false);
  pp_key_stringvalue(stream, 2, "int_binary_filename", i->int_binary_filename,false);
  pp_key_stringvalue(stream, 2, "int_first_fields", i->int_first_fields,false);
  pp_key_stringvalue(stream, 2, "int_last_fields", i->int_last_fields,true);
  pp_end(stream,0,last);
}



/* ******* */

void pp_skip(FILE*stream, string_list_t ss) {
  //  printf("SKIP");
  if (ss==0) 
    return;
  else {
    pp_key_stringvalue(stream, 2, "skip", ss->string,false);
    pp_skip(stream,ss->next);
    return;
  }
}

void pp_expectations(FILE*stream, string_pair_list_t sps) {
  //  printf("EXPECTATIONS");
  if (sps==0) 
    return;
  else {
    pp_keystring_stringvalue(stream, 2, sps->string1, sps->string2,false);
    pp_expectations(stream,sps->next);
    return;
  }
}

void pp_test(FILE*stream, test_t t) {
  pp_start(stream);
  pp_key_stringvalue(stream, 2, "test", t->src_main,false);
  pp_skip(stream, t->skip);
  pp_expectations(stream, t->expectations);
  pp_end(stream,0,false);
}

 void pp_tests_body(FILE*stream, test_list_t ts) {
   if (ts==0) 
     return;
   else {
     pp_test(stream, ts->test);
     if (ts->next!=0) printf(",");
     pp_tests_body(stream,ts->next);
   }
 }


 void pp_tests(FILE*stream, test_list_t ts) {
   printf("[\n");
   pp_tests_body(stream,ts);
   printf("]\n");
 }




 void pp_tools_body(FILE*stream, tool_list_t ts) {
   if (ts==0) 
     return;
   else {
     pp_tool(stream, 2, &ts->tool, false);
     if (ts->next!=0) printf(",");
     pp_tools_body(stream,ts->next);
   }
 }


 void pp_tools(FILE*stream, tool_list_t ts) {
   printf("[\n");
   pp_tools_body(stream,ts);
   printf("]\n");
 }



/* ***************************************************************** */
/*  reading tests.charon                                             */
/* ***************************************************************** */

static int dump(const char *js, jsmntok_t *tbase, jsmntok_t *t, size_t count, int indent) {
	int i, j, k;
	if (count == 0) {
		return 0;
	}
	if (t->type == JSMN_PRIMITIVE) {
          printf("%li:P %.*s", t-tbase, t->end - t->start, js+t->start);
		return 1;
	} else if (t->type == JSMN_STRING) {
          printf("%li:S '%.*s'", t-tbase, t->end - t->start, js+t->start);
		return 1;
	} else if (t->type == JSMN_OBJECT) {
          printf("%li:O \n",t-tbase);
		j = 0;
		for (i = 0; i < t->size; i++) {
			for (k = 0; k < indent; k++) printf("  ");
			j += dump(js, tbase, t+1+j, count-j, indent+1);
			printf(": ");
			j += dump(js, tbase, t+1+j, count-j, indent+1);
			printf("\n");
		}
		return j+1;
	} else if (t->type == JSMN_ARRAY) {
		j = 0;
		printf("%li:A \n",t-tbase);
		for (i = 0; i < t->size; i++) {
			for (k = 0; k < indent-1; k++) printf("  ");
			printf("   - ");
			j += dump(js, tbase, t+1+j, count-j, indent+1);
			printf("\n");
		}
		return j+1;
	}
	return 0;
}


static int json_next(const char *js, jsmntok_t *t, int j) {
  int i, count;
  if (t[j].type == JSMN_PRIMITIVE) {
    return j+1;
  } else if (t[j].type == JSMN_STRING) {
    return j+1;
  } else if (t[j].type == JSMN_OBJECT) {
    count = t[j].size;
    j = j + 1;
    for (i = 0; i < count; i++) {
      j = json_next(js, t, j);
      j = json_next(js, t, j);
    }
    return j;
  } else if (t[j].type == JSMN_ARRAY) {
    count = t[j].size;
    j = j + 1;
    for (i = 0; i < count; i++) {
      j = json_next(js, t, j);
    }
    return j;
  }
  return 0;
}



void dump_simple(const char *js, jsmntok_t *t, size_t count) {
  int i;
  for (i=0; i<count; i++) {
    //printf("i=%i:",i);
    if (t[i].type == JSMN_PRIMITIVE) {
      printf("%i:P[%d,%d] %.*s", i,t[i].start, t[i].end, t[i].end - t[i].start, js+t[i].start);
    } else if (t[i].type == JSMN_STRING) {
      printf("%i:S[%d,%d] '%.*s'", i,t[i].start, t[i].end, t[i].end - t[i].start, js+t[i].start);
    } else if (t[i].type == JSMN_OBJECT) {
      printf("\n%i:O(%d)[%d,%d]",i,t[i].size, t[i].start, t[i].end);
    } else if (t[i].type == JSMN_ARRAY) {
      printf("\n%i:A(%d)[%d,%d]",i,t[i].size, t[i].start, t[i].end);
    } else {
      printf("ERR");
    };
    printf("  ");
  }
  printf("\n");
  return;
}



// ghastly hack to return a tuple
int tok_count;
char *buffer;


void parse_error(const char *s, int pos) {
  fprintf(stderr, "parse error: %s at character %d\n",s,pos);
  exit(EXIT_FAILURE);
}


jsmntok_t *read_json(char *filename) {
  int r;
  
  jsmn_parser p;
  jsmntok_t *tok;
  size_t tokcount = 200;

  /* read file */ 
  string_t s;
  s = string_from_file(filename);
  
  /* Prepare parser */
  jsmn_init(&p);

  /* Allocate some tokens as a start */
  tok = malloc(sizeof(*tok) * tokcount);
  if (tok == NULL) {
    fprintf(stderr, "malloc(): errno=%d\n", errno);
    exit(EXIT_FAILURE);
  }
   
 again:
  r = jsmn_parse(&p, s->buffer, s->data_length, tok, tokcount);
  if (r < 0) {
    if (r == JSMN_ERROR_NOMEM) {
      tokcount = tokcount * 2;
      tok = realloc(tok, sizeof(*tok) * tokcount);
      if (tok == NULL) {
        fprintf(stderr, "realloc(): errno=%d\n", errno);
        exit(EXIT_FAILURE);
      }
      goto again;
    }
    else if (r == JSMN_ERROR_INVAL) 
      parse_error("jsmn_parse(): invalid character in JSON string",p.pos);
    else if (r == JSMN_ERROR_PART)
      parse_error("jsmn_parse(): string not a full JSON packet, more bytes expected",p.pos);
    else 
      parse_error("jsmn_parse(): unknown error",p.pos);
  } else {
    //dump_simple(s->buffer, tok, r);
    //dump(s->buffer, tok, tok, r, 0);
    tok_count = r;   // ghastly imperative hack
    buffer = s->buffer;
    return(tok);
  }
  return(0);
}


int read_test(char *buffer, jsmntok_t *tok, int j, test_list_t *ts) {

  test_t t;
  test_list_node_t *tn;
  t = (test_t)malloc(sizeof(struct test_s));
  tn = (test_list_node_t*)malloc(sizeof(test_list_node_t));
  t->src_main=0;
  t->src_aux=0;
  t->skip=0;
  t->expectations=0;
  t->compile_can_fail = false;
  tn->test = t;
  tn->next=*ts;

  if (tok[j].type != JSMN_OBJECT) 
    parse_error("read_test: not an object",tok[j].start);
  
  int count;
  count = tok[j].size;   
  //  printf("number of fields = %i\n", count);
  j++;
 
  while (count >=1 ) {

    if ((tok[j].type == JSMN_PRIMITIVE || tok[j].type == JSMN_STRING) && memcmp(buffer+tok[j].start,"test",4)==0) {
      if (t->src_main !=0) 
        parse_error("repeated test name", tok[j].start);
      if (tok[j+1].type != JSMN_PRIMITIVE && tok[j+1].type != JSMN_STRING) 
        parse_error("test name not a primitive or string", tok[j+1].start);
      t->src_main = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
      j = j+2;
    } else if ((tok[j].type == JSMN_PRIMITIVE || tok[j].type == JSMN_STRING) && memcmp(buffer+tok[j].start,"src_aux",strlen("src_aux"))==0) {
      if (tok[j+1].type != JSMN_PRIMITIVE && tok[j+1].type != JSMN_STRING)
        parse_error("extra_src not a primitive or string",tok[j+1].start);
      string_list_t sl = (string_list_t)malloc(sizeof(string_list_node_t));
      sl->string = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
      printf("FOUND SRCS_AUX: %s\n", sl->string->buffer);
      sl->next = t->src_aux;
      t->src_aux = sl;
      j = j+2;
    } else if ((tok[j].type == JSMN_PRIMITIVE || tok[j].type == JSMN_STRING) && memcmp(buffer+tok[j].start,"skip",4)==0) {
      if (tok[j+1].type != JSMN_PRIMITIVE && tok[j+1].type != JSMN_STRING) 
        parse_error("skip not a primitive or string",tok[j+1].start);
      string_list_t sl = (string_list_t)malloc(sizeof(string_list_node_t));
      sl->string = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
      sl->next = t->skip;
      t->skip = sl;
      j = j+2;
    } else if (tok[j].type == JSMN_PRIMITIVE || tok[j].type == JSMN_STRING) {
      if (tok[j+1].type != JSMN_PRIMITIVE && tok[j+1].type != JSMN_STRING) 
        parse_error("non-main/skip not a primitive or string",tok[j+1].start);
      string_t expectation_target = string_from_memory(buffer+tok[j].start, buffer+tok[j].end);
      string_t expectation_body = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
      string_pair_list_t spl;
      spl = (string_pair_list_t)malloc(sizeof(string_pair_list_node_t));
      spl->string1 = expectation_target;
      spl->string2 = expectation_body;
      spl->next = t->expectations;
      t->expectations = spl;
      if (memcmp(expectation_target->buffer, "compile_can_fail", strlen("compile_can_fail")) == 0) {
        printf("COMPILE CAN FAIL FOR: %s\n", t->src_main->buffer);
        t->compile_can_fail = true;
      }
      j = j+2;
    } else
      parse_error("bad field", tok[j].start);

    count--;
  }
  assert(t->src_main->buffer != NULL);
  *ts = tn;
  return(j);
}
  


test_list_t read_tests(char *test_descriptions_filename) {
  jsmntok_t *tok;
  tok = read_json(test_descriptions_filename);

  test_list_t ts;
  ts = 0;

  if (!(tok_count >=1 && tok[0].type == JSMN_ARRAY)) 
    parse_error("read_tests: not an array of nonzero size", tok[0].start);
  
  int number_of_tests;
  number_of_tests = tok[0].size;
  //printf("number of tests = %i\n", number_of_tests);
  int i,j;

  j = 1;
  for (i = 0; i < number_of_tests; i++) {
    j = read_test(buffer, tok, j, &ts);
  }


  return(ts);

}

static bool is_key_name(const char* buffer, jsmntok_t *tok, const char* name) {
  int maxlen = tok->end - tok->start;
  if (strlen(name) < maxlen)
    maxlen = strlen(name);
  if (tok->type == JSMN_PRIMITIVE && memcmp(buffer + tok->start, name, maxlen) == 0)
    return true;
  if (tok->type == JSMN_STRING && memcmp(buffer + tok->start, name, maxlen) == 0)
    return true;
  return false;
}

/* ***************************************************************** */
/*  reading tools.charon                                             */
/* ***************************************************************** */


int read_tool(char *buffer, jsmntok_t *tok, int j, tool_list_t *ts) {
  //printf("read_tool j=%i\n",j);
  tool_list_node_t *tn;
  tn = (tool_list_node_t*)malloc(sizeof(tool_list_node_t));
  tn->next=*ts;

  if (tok[j].type != JSMN_OBJECT) 
    parse_error("read_tool: not an object", tok[j].start);
  
  int count;
  count = tok[j].size;   
  //  printf("number of fields = %i\n", count);
  if ((count !=3) && (count !=4)) 
    parse_error("tool entry does not have 3 or 4 fields",tok[j].start);

  j++;
 
  
  if (is_key_name(buffer, &tok[j], "cmd") && (tok[j+1].type == JSMN_PRIMITIVE || tok[j+1].type == JSMN_STRING))
    tn->tool.tool_name = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
  else 
    parse_error("ill-formed tool cmd",tok[j].start);

    
  j = j+2;
  if (is_key_name(buffer, &tok[j], "args") && (tok[j+1].type == JSMN_PRIMITIVE || tok[j+1].type == JSMN_STRING))
    tn->tool.tool_args = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
  else 
    parse_error("ill-formed tool args", tok[j].start);

  j = j + 2;
  if (is_key_name(buffer, &tok[j], "name") && (tok[j+1].type == JSMN_PRIMITIVE || tok[j+1].type == JSMN_STRING))
    tn->tool.tool_instance_name = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
  else 
    parse_error("ill-formed tool instance name", tok[j].start);

  if (count == 4) {
    j = j + 2;
    if (is_key_name(buffer, &tok[j], "single_phase") && (tok[j+1].type == JSMN_PRIMITIVE || tok[j+1].type == JSMN_STRING)) {
      if (memcmp(buffer+tok[j+1].start,"true",4)==0 )
        tn->tool.tool_single_phase = true;
      else if (memcmp(buffer+tok[j+1].start,"false",5)==0 )
        tn->tool.tool_single_phase = false;
      else 
        parse_error("ill-formed tool single_phase body", tok[j+1].start);
    } else  if (is_key_name(buffer, &tok[j], "run_prefix") && (tok[j+1].type == JSMN_PRIMITIVE || tok[j+1].type == JSMN_STRING)) {
      tn->tool.tool_run_prefix = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);
    } else { 
      parse_error("ill-formed tool 4th field, not single_phase", tok[j].start);
    }
  } else {
    tn->tool.tool_single_phase = false;
    tn->tool.tool_run_prefix = NULL;
  }

  j = j + 2;
  *ts = tn;
  return(j);
}
  


tool_list_t read_tools(char *tool_descriptions_filename) {
  jsmntok_t *tok;
  tok = read_json(tool_descriptions_filename);

  tool_list_t ts;
  ts = 0;

  if (!(tok_count >=1 && tok[0].type == JSMN_ARRAY)) 
    parse_error("read_tools: not an array of nonzero size", tok[0].start);
  
  int number_of_tools;
  number_of_tools = tok[0].size;
  //printf("number of tools = %i\n", number_of_tools);
  int i,j;

  j = 1;
  for (i = 0; i < number_of_tools; i++) {
    j = read_tool(buffer, tok, j, &ts);
  }

  
  return(ts);

}



/* ***************************************************************** */
/*  read intermediate log file                                       */
/* ***************************************************************** */


int read_intermediate(char *buffer, jsmntok_t *tok, int j_init, int_list_t *is) {

  int j;
  j = j_init;

  int_t i;
  int_list_node_t *in;
  i = (int_t)malloc(sizeof(struct int_s));
  in = (int_list_node_t*)malloc(sizeof(int_list_node_t));
  i->int_test_instance_name=NULL;
  i->int_binary_filename=NULL;
  i->int_first_fields=NULL;
  i->int_last_fields=NULL;
  in->intv = i;
  in->next=*is;

  if (tok[j].type != JSMN_OBJECT) 
    parse_error("read_intermediate: not an object",tok[j].start);

  int execute_output_start;
  int execute_output_end;
  execute_output_start=0;
  execute_output_end=0;

  int count;
  count = tok[j].size;   
  //  printf("number of fields = %i\n", count);

  j = j+1; // advance to first field of this object
 
  while (count >=1 ) {
    // printf("count = %d  j=%d*****************\n",count,j);
 
    // find test_instance_name field value of test_recipe field 
    if (is_key_name(buffer, &tok[j], "test_recipe")) {
      if (i->int_test_instance_name !=NULL) 
        parse_error("repeated test_instance_name", tok[j].start);

      if (tok[j+1].type != JSMN_OBJECT) 
        parse_error("test_recipe not an object", tok[j+1].start);

      if (! (is_key_name(buffer, &tok[j+2], "test_instance_name")))
        parse_error("first field of test_recipe not test_instance_name", tok[j+2].start);
      if (! ((tok[j+3].type == JSMN_PRIMITIVE || tok[j+3].type == JSMN_STRING)))
        parse_error("test_instance_name not primitive or string", tok[j+3].start);

      //printf("TEST_INSTANCE_NAME*****************\n");

      i->int_test_instance_name = string_from_memory(buffer+tok[j+3].start, buffer+tok[j+3].end);

    // find binary_filename field
    } else if (is_key_name(buffer, &tok[j], "binary_filename")) {
      if ( !((tok[j+1].type == JSMN_PRIMITIVE || tok[j+1].type == JSMN_STRING)))
        parse_error("value of binary_filename not a primitive or string",tok[j+1].start);
      i->int_binary_filename = string_from_memory(buffer+tok[j+1].start, buffer+tok[j+1].end);

    // find execute_host and execute_output fields, to identify the first- and last- fields before and after them
    } else if (is_key_name(buffer, &tok[j], "execute_host")) {
      //printf("EXECUTE*****************\n");
      if ( !((tok[j+1].type == JSMN_PRIMITIVE || tok[j+1].type == JSMN_STRING) && memcmp(buffer+tok[j+1].start,"NULL",4)==0)) 
        parse_error("value of execute_host not a primitive or string containing \"NULL\"",tok[j+1].start);
      if ( !(is_key_name(buffer, &tok[j+2], "execute_output")))
        parse_error("successor of execute_host not an execute_output field",tok[j+2].start);
      if ( !((tok[j+3].type == JSMN_PRIMITIVE || tok[j+3].type == JSMN_STRING) && memcmp(buffer+tok[j+3].start,"NULL",4)==0)) 
        parse_error("value of execute_output not a primitive or string containing \"NULL\"",tok[j+3].start);

      execute_output_start = tok[j].start;
      execute_output_end = tok[j+3].end;
     

     
    // skip over any other field 
    } else {
    }

    j = json_next(buffer, tok, j); 
    //printf("j=%d\n",j);
    j = json_next(buffer, tok, j);
    //printf("j=%d\n\n",j);
    count--;
  }
  if (i->int_test_instance_name == NULL) parse_error("no test_instance_name found",tok[j_init].start);
  if (i->int_binary_filename == NULL) parse_error("no binary_filename found",tok[j_init].start);
  if (execute_output_start == 0 || execute_output_end==0) parse_error("no execute_output field found",tok[j_init].start);


  i->int_first_fields = string_from_memory(buffer + tok[j_init].start, buffer + execute_output_start -1 );
  i->int_last_fields  = string_from_memory(buffer + execute_output_end + 1, buffer + tok[j_init].end);

  *is = in;
  
  //pp_int(stdout, i, true);

  return(j);
}
 

int_list_t int_list_reverse(int_list_t head) {
  int_list_node_t *prev = NULL;
  int_list_node_t *next;
  
  while (head) {
    next = head->next;
    head->next = prev;
    prev = head;
    head = next;
  }
  
  return prev;
} 


int_list_t read_intermediates(char *int_filename) {
  jsmntok_t *tok;
  tok = read_json(int_filename);
  int_list_t is;
  is = NULL;
  if (!(tok_count >=1 && tok[0].type == JSMN_ARRAY)) 
    parse_error("read_intermediates: not an array of nonzero size", tok[0].start);
  int number_of_ints;
  number_of_ints = tok[0].size;
  //printf("number of ints = %i\n", number_of_ints);
  int i,j;
  j = 1; // advance to first object of array
  for (i = 0; i < number_of_ints; i++) {
    j = read_intermediate(buffer, tok, j, &is);
  }
  return(int_list_reverse(is));
}


/* ***************************************************************** */
/*  sample test recipe                                               */
/* ***************************************************************** */

recipe_t test_recipe() {
  recipe_t r;
  r = malloc(sizeof(struct recipe_s));
  sources_t sources;
  sources = malloc(sizeof(struct sources_s));
  tool_t tool;
  tool = malloc(sizeof(struct tool_s));
  r->sources=sources;
  r->tool=tool;
  tool->tool_name = string_from_constant("gcc-4.8");
  tool->tool_args = string_from_constant("-O2 -std=c11 -pedantic -Wall -Wextra -Wno-unused-variable -pthread");
  tool->tool_instance_name = string_from_constant("gcc-4.8");
  sources->src_dir = string_from_constant("../notes/examples/");
  sources->src_main = string_from_constant("pointer_representation_6.c");
  sources->src_aux = NULL;
  return(r);
} 


/* ***************************************************************** */
/*  construction of test recipes from tests.charon x tools.charon    */
/* ***************************************************************** */

bool string_substring(string_t s1, string_t s2) {
  //printf("DEBUG: s1=");  pp_string(stdout, s1); printf("\n");
  //printf("DEBUG: s2=");  pp_string(stdout, s2); printf("\n");
  if (s1->data_length > s2->data_length) {
    return false;
  }
  int offset;
  for (offset=0; offset <= s2->data_length - s1->data_length; offset++) 
    if (0==memcmp(s1->buffer, s2->buffer + offset, s1->data_length)) {
      return true;
    }
  return false;
}

bool skip(string_list_t skip_list, string_t tool_instance_name) {
  if (skip_list==0) 
    return(false);
  else
    return(string_substring(skip_list->string, tool_instance_name) || skip(skip_list->next,tool_instance_name));
}

recipe_t make_test_recipe(string_t test_dir, test_t test, tool_t tool) {
  recipe_t r;
  r = malloc(sizeof(struct recipe_s));
  r->test_instance_name = app(string_const_append(test->src_main, "."), tool->tool_instance_name); 
  sources_t sources;
  sources = malloc(sizeof(struct sources_s));
  r->sources=sources;
  sources->src_dir = test_dir; 
  sources->src_main = test->src_main;
  sources->src_aux = test->src_aux;
  sources->compile_can_fail = test->compile_can_fail;
  r->tool=tool;
  return(r);
}

recipe_list_t make_test_recipes_aux(string_t test_dir, test_t test, tool_list_t tls, recipe_list_t rs) {
  if (tls==0) 
    return(rs);
  else {
    tool_t tool;
    tool = &(tls->tool);
    if (skip(test->skip, tool->tool_instance_name)) 
      return(make_test_recipes_aux(test_dir,test,tls->next,rs));
    else {
      recipe_t r;
      r = make_test_recipe(test_dir, test, tool);
      recipe_list_t rn;
      rn = (recipe_list_t)malloc(sizeof(struct recipe_list_node_s));
      rn->recipe=r;
      rn->next=rs;
      return(make_test_recipes_aux(test_dir,test,tls->next,rn));
    } 
  }
}
    
  



recipe_list_t make_test_recipes(string_t test_dir, test_list_t ts, tool_list_t tls, recipe_list_t rs) {
  if (ts==0) 
    return(rs);
  else {
    test_t test;
    test = ts->test;
    recipe_list_t rs2;
    rs2=make_test_recipes_aux(test_dir,test,tls,rs);
    return(make_test_recipes(test_dir,ts->next,tls,rs2));
  }
} 

int length_recipe_list(recipe_list_t rs) {
  if (rs==0) 
    return(0);
  else
    return(1+length_recipe_list(rs->next));
}

int length_int_list(int_list_t is) {
  if (is==0)
    return(0);
  else
    return(1+length_int_list(is->next));
}




/* ***************************************************************** */
/*  running a test                                                   */
/* ***************************************************************** */

string_t mk_binary_filename(string_t test_bin_dir, string_t test_instance_name) {
  string_t n;
  n = app(test_bin_dir, 
          app(test_instance_name,
              string_from_constant(".out")));
 return n;
}


string_t run_command_to_string(string_t cmd) {
  string_t cmd2;
//  printf("DEBUG:");
//  pp_string(stdout, cmd);
//  printf("\n");
  cmd2 = string_const_append_term(cmd, " 1> tmp/tmp.charon.stdout");
  int status;
  
  // run command
  status = system(cmd2->buffer);
  if (status == -1) perror_exit("system returned -1\n");

  // read stdout string
  return (string_from_file("tmp/tmp.charon.stdout"));
}

// variant of run_command_to_string that strips trailing newline if present
string_t run_command_to_string_nonl(string_t cmd) {
  string_t s;
  s = run_command_to_string(cmd);
  if (s->data_length>=1 && s->buffer[s->data_length-1] == '\n') 
    s->data_length = s->data_length-1;
  return(s);
}

output_t run_test_to_output(string_t cmd) {
  output_t o = malloc(sizeof(struct output_s));
  int status;
  struct timeval tv_start;

  // save start time
  status = gettimeofday(&tv_start, NULL);

  // run command
  pid_t child = vfork();
  if (child == 0)
  {
    char *args[] = {cmd->buffer, NULL};
    close(1);
    int out = open("tmp/tmp.charon.stdout", O_WRONLY | O_CREAT | O_TRUNC, 0644);
    if (out != 1)
    {
      perror("Failed to open standard output file");
      _exit(EXIT_FAILURE);
    }
    close(2);
    int err = open("tmp/tmp.charon.stderr", O_WRONLY | O_CREAT | O_TRUNC, 0644);
    if (err != 2)
    {
      perror("Failed to open standard error file");
      _exit(EXIT_FAILURE);
    }
    // Run the command, with no arguments, inheriting our environment
    execv(cmd->buffer, args);
    // If we get here, execve failed.  Notify the parent
    _exit(EXIT_FAILURE);
  }
  struct rusage rusage;
  if (wait4(child, &status, 0, &rusage) != child)
  {
    perror_exit("Failed to get exit status of child");
  }
  o->duration = rusage.ru_utime;

  // copy command to output
  o->command = cmd;
  
  // read stdout and stderr strings and populate output
  o->command_stdout = string_from_file("tmp/tmp.charon.stdout");
  o->command_stderr = string_from_file("tmp/tmp.charon.stderr");


  // populate output with signals string 
  if (WIFSIGNALED(status))
  {
    printf("SIGNALED\n");
    int signo = WTERMSIG(status);
    o->exit_code = -1;
    o->signals = malloc(sizeof(struct string_s));
    o->signals->data_length = asprintf(&o->signals->buffer,
        "Terminated with signal %d: %s", signo, strsignal(signo));
    o->signals->buffer_length = o->signals->data_length;
  }
  else
  {
    assert(WIFEXITED(status));
    o->exit_code = WEXITSTATUS(status);
    o->signals = NULL;
  }

  // pretty print start time in localtime and populate output
  struct tm* t; 
  t = localtime(&tv_start.tv_sec);
  int time_string_max_length = 80;
  o->start_time = string_create(time_string_max_length);
  int len;
  len = snprintf(o->start_time->buffer, time_string_max_length, 
                 "%4i.%02i.%02i %i:%02i:%02i.%06ld" , 1900+t->tm_year, 1+t->tm_mon, t->tm_mday, 
                 t->tm_hour, t->tm_min, t->tm_sec,
                 (long)tv_start.tv_usec);
  if (len >= time_string_max_length) myerror_exit("snprintf\n");
  o->start_time->data_length = len;

  return(o);
}

output_t run_command_to_output(string_t cmd) {
  output_t o = malloc(sizeof(struct output_s));
  string_t cmd2;
  cmd2 = string_const_append_term(cmd, " 1> tmp/tmp.charon.stdout 2> tmp/tmp.charon.stderr");
  int status, tod;
  struct timeval tv_start, tv_end;

  // save start time
  status = gettimeofday(&tv_start, NULL);
  if (status == -1) perror_exit("gettimeofday returned -1\n");
  
  // run command
  status = system(cmd2->buffer);
  if (status == -1) perror_exit("system returned -1\n");

  // save end time
  tod = gettimeofday(&tv_end, NULL);
  if (tod == -1) perror_exit("gettimeofday returned -1\n");

  // copy command to output
  o->command = string_strip_term(cmd2);
  o->signals = NULL;
  
  // read stdout and stderr strings and populate output
  o->command_stdout = string_from_file("tmp/tmp.charon.stdout");
  o->command_stderr = string_from_file("tmp/tmp.charon.stderr");

  // populate output with exit code
  o->exit_code = WEXITSTATUS(status);

  // pretty print start time in localtime and populate output
  struct tm* t; 
  t = localtime(&tv_start.tv_sec);
  int time_string_max_length = 80;
  o->start_time = string_create(time_string_max_length);
  int len;
  len = snprintf(o->start_time->buffer, time_string_max_length, 
                 "%4i.%02i.%02i %i:%02i:%02i.%06ld" , 1900+t->tm_year, 1+t->tm_mon, t->tm_mday, 
                 t->tm_hour, t->tm_min, t->tm_sec,
                 (long)tv_start.tv_usec);
  if (len >= time_string_max_length) myerror_exit("snprintf\n");
  o->start_time->data_length = len;

  // calculate time delta and populate output
  timersub(&tv_end, &tv_start, &o->duration);


  return(o);
}

host_t get_host() {
  string_t get_host_name = string_from_constant("hostname");
#ifdef __APPLE__
  string_t get_host_cpuinfo = string_from_constant("sysctl -n machdep.cpu.brand_string");
#elif __FreeBSD__
  string_t get_host_cpuinfo = string_from_constant("sysctl -n hw.model");
#else
  string_t get_host_cpuinfo = string_from_constant("cat /proc/cpuinfo | grep \"model name\" | head -1");
#endif
  string_t get_host_os = string_from_constant("uname -s");
  string_t get_host_os_version = string_from_constant("uname -v -r");
  string_t get_host_hardware_platform = string_from_constant("uname -m");
  host_t h;
  h = malloc(sizeof(struct host_s));
  h->host_name = run_command_to_string_nonl(get_host_name);
  h->host_os = run_command_to_string_nonl(get_host_os);
  h->host_os_version = run_command_to_string_nonl(get_host_os_version);
  h->host_hardware_platform = run_command_to_string_nonl(get_host_hardware_platform);
  h->host_cpuinfo = run_command_to_string_nonl(get_host_cpuinfo);
  return(h);
}
 
// TODO: multi-file tests not implemented yet
observation_t run_test(host_t h, recipe_t r, string_t test_bin_dir, execution_mode_t mode) {

  observation_t o;
  o = malloc(sizeof(struct observation_s));
  o->test_recipe = r;
  o->test_name = r->sources->src_main; 
  o->tool_version = run_command_to_string_nonl(string_const_append(r->tool->tool_name, " --version | head -1"));
  string_t srcs;
  srcs = app(r->sources->src_dir, r->sources->src_main);

  
//  if (string_prefix(string_from_constant("cerberus"), r->tool->tool_name)) {
//    /* special case for Cerberus: elaborate and execute */
//    if (! (mode==COMPILE_AND_RUN)) {
//      printf("Error: Cerberus is only supported in compile-and-run mode, not in run-only mode\n");
//      exit(EXIT_FAILURE);
//    }
//    o->compile_host = h;
//    o->compile_output = 
//      run_command_to_output(
//                            app(r->tool->tool_name,
//                                app(string_from_constant(" "), 
//                                    app(r->tool->tool_args, 
//                                        srcs))));
//    o->binary_filename = NULL;
//    o->execute_host = h;
//    o->execute_output =
//      run_command_to_output(
//                            app(r->tool->tool_name,
//                                app(string_from_constant(" --exec "), 
//                                    app(r->tool->tool_args, 
//                                        srcs))));
//      
//  } else 

  if (mode==COMPILE_ONLY && r->tool->tool_single_phase) {
    printf("Error: COMPILE_ONLY mode not compatible with a single-phase tool\n");
    exit(EXIT_FAILURE);
  }

  if (mode==COMPILE_AND_RUN && r->tool->tool_single_phase) {
    /* single-phase tool: just execute */
    o->compile_host = NULL;
    o->compile_output = NULL;
    o->compile_output_aux = NULL;
    o->binary_filename = NULL;
    o->execute_host = h;
    o->execute_output =
      run_command_to_output(
                            app(r->tool->tool_name,
                                app(string_from_constant(" "), 
                                    app(r->tool->tool_args, 
                                        app(string_from_constant(" "),
                                            srcs)))));

  } else {
    /* normal compiler: compile and run  or compile-only*/

    string_t test_binary;
    test_binary = mk_binary_filename(test_bin_dir, r->test_instance_name);
    o->compile_host = h;
    // For multi-source tests compile separate .o files to avoid
    // cross-translation-unit optimizations
    string_list_t src_aux = r->sources->src_aux;
    string_t compiler_srcs = srcs;
    while (src_aux) {
      string_t aux_srcfile = app(r->sources->src_dir, src_aux->string);
      string_t aux_objfile = app(test_bin_dir,
                                 app(src_aux->string,
                                     string_from_constant(".o")));
      o->compile_output_aux =
        run_command_to_output(
                              app(string_from_constant("LANG=C "),
                                  app(r->tool->tool_name,
                                      app(string_from_constant(" "),
                                          app(r->tool->tool_args,
                                              app(string_from_constant(" -c -o "),
                                                  app(aux_objfile,
                                                      app(string_from_constant(" "),
                                                          aux_srcfile))))))));
      // Add the auxilliary .o file to the final compile command line
      compiler_srcs = app(compiler_srcs, app(string_from_constant(" "), aux_objfile));
      // And the source .c file to the files to be hashed and added to the json
      srcs = app(srcs, app(string_from_constant(" "), aux_srcfile));
      printf("Building auxilliary source with command: %.*s\n",
             PRINTF_STRING_T_ARG(o->compile_output_aux->command));
      if (o->compile_output_aux->exit_code != 0) {
        fprintf(stderr, "Failed to compile auxilliary source %.*s!\n", PRINTF_STRING_T_ARG(aux_srcfile));
        fprintf(stderr, "Command was: %.*s\n", PRINTF_STRING_T_ARG(o->compile_output_aux->command));
        fprintf(stderr, "%.*s\n", PRINTF_STRING_T_ARG(o->compile_output_aux->command_stderr));
        if (!r->sources->compile_can_fail)
          exit(EXIT_FAILURE);
      }
      src_aux = src_aux->next;
    }

    o->compile_output = 
      run_command_to_output(
                            app(string_from_constant("LANG=C "),
                                app(r->tool->tool_name,
                                    app(string_from_constant(" "), 
                                        app(r->tool->tool_args, 
                                            app(string_from_constant(" -o "),
                                                app(test_binary,
                                                    app(string_from_constant(" "),
                                                        compiler_srcs))))))));
    if (r->sources->src_aux)
      printf("Built multi source test with command: %.*s\n", PRINTF_STRING_T_ARG(o->compile_output->command));
    o->binary_filename = test_binary;
    if (mode==COMPILE_ONLY && o->compile_output->exit_code != 0) {
      fprintf(stderr, "Failed to compile %.*s!\n", PRINTF_STRING_T_ARG(o->binary_filename));
      fprintf(stderr, "Command was: %.*s\n", PRINTF_STRING_T_ARG(o->compile_output->command));
      fprintf(stderr, "%.*s\n", PRINTF_STRING_T_ARG(o->compile_output->command_stderr));
      if (!r->sources->compile_can_fail) {
        /* FIXME: currently the CHERI tests are broken if an address needs to be guessed */
        const char* ignore_compile_failures = getenv("IGNORE_COMPILATION_FAILURES");
        if (ignore_compile_failures == NULL || *ignore_compile_failures == '\0') {
          exit(EXIT_FAILURE);
        }
      }
    }
    if (mode == COMPILE_AND_RUN) {
      o->execute_host = h;
      if (r->tool->tool_run_prefix == NULL) 
        o->execute_output =   
          run_command_to_output( test_binary );
      else
        o->execute_output =   
          run_command_to_output( 
                                app(r->tool->tool_run_prefix, 
                                    app(string_from_constant(" "),
                                        test_binary )));
    }
    else if (mode == COMPILE_ONLY) {
      o->execute_host = NULL;
      o->execute_output = NULL;
    }
    else {
      printf("Error: unexpected mode in run_test\n");
      exit(EXIT_FAILURE);
    }
  }
  o->source_hashes = 
    run_command_to_string(
                          app(
#ifndef __linux__
                              string_from_constant("md5 "),
#else
                              string_from_constant("md5sum -t "),
#endif
                              //string_from_constant("openssl dgst -sha1 "),
                              srcs));
  o->sources = 
    run_command_to_string(
                          app(
                              string_from_constant("cat "),
                              srcs));
  //o->assembly = NULL;
  //o->binary = NULL;
  return(o); 
  
}

void log_observation(observation_t o, FILE *main_log_file, string_t test_log_dir, bool last) {
  // log to main log file 
  pp_observation(main_log_file, o, last);

  // log to per-test log file
  FILE *test_log_file;
  string_t log_filename;
  log_filename = string_const_append_term(
                                            app(test_log_dir,
                                                o->test_recipe->test_instance_name),
                                            ".log");

  test_log_file = fopen(log_filename->buffer,"w");
  if (test_log_file == NULL) {
    fprintf(stderr, "fopen of %s: errno=%d\n",log_filename->buffer, errno);
    exit(EXIT_FAILURE);
  }
  pp_observation(test_log_file, o, true);
  fclose(test_log_file);
  return;
}



output_t run_test_from_intermediate(int_t i) {
  output_t o;
  o = run_test_to_output( i->int_binary_filename );
  return o;
}


void log_observation_from_intermediate(host_t h, int_t i, output_t o, FILE *main_log_file, string_t test_log_dir, bool last) {
  // log to main log file 
  pp_observation_from_intermediate(main_log_file, h, i, o, last);

  // log to per-test log file
  FILE *test_log_file;
  string_t log_filename;
  log_filename = string_const_append_term(
                                            app(test_log_dir,
                                                i->int_test_instance_name),
                                            ".log");

  test_log_file = fopen(log_filename->buffer,"w");
  if (test_log_file == NULL) {
    fprintf(stderr, "fopen of %s: errno=%d\n",log_filename->buffer, errno);
    exit(EXIT_FAILURE);
  }
  pp_observation_from_intermediate(test_log_file, h, i, o, true);
  fclose(test_log_file);
  return;
}




/* ***************************************************************** */
/*  process command-line options                                     */
/* ***************************************************************** */

execution_mode_t mode = COMPILE_AND_RUN;
char *tests_filename = 0;
char *tools_filename = 0;
char *log_filename = 0;
char *int_filename = 0;
string_t test_src_dir;
string_t test_bin_dir;
string_t test_int_dir;
string_t test_log_dir;

void process_command_line_options(int argc, char **argv) {
  /*
  char* tools_filename_default = "tools.charon";
  char* tests_filename_default = "tests.charon";
  char* test_src_dir_filename_default = "tests/de_facto_memory_model/";
  char* test_bin_dir_filename_default = "tests.bin/";
  char* test_int_dir_filename_default = "tests.int/";
  char* test_log_dir_filename_default = "tests.log/";
  char* log_filename_default = "tests.log/all.log";
  char* int_filename_default = "tests.int/all.log";
  */

  int c;
  char *test_src_dir_filename = 0;
  char *test_bin_dir_filename = 0;
  char *test_int_dir_filename = 0;
  char *test_log_dir_filename = 0;
  bool compile_only = false;
  bool run_only = false;

  /* process command-line arguments */
  while (1)
    {
      static struct option long_options[] =
        {
          {"compile_and_run",   no_argument, 0, 'A'},
          {"compile_only",   no_argument, 0, 'c'},
          {"run_only",   no_argument, 0, 'r'},
          {"tools",   required_argument, 0, 'o'},
          {"tests",   required_argument, 0, 'e'},
          {"test_src_dir",required_argument, 0, 's'},
          {"test_bin_dir",required_argument, 0, 'b'},
          {"test_int_dir",required_argument, 0, 'i'},
          {"test_log_dir",required_argument, 0, 'l'},
          {"log",     required_argument, 0, 'L'},
          {"int",     required_argument, 0, 'I'},
          {"help",    no_argument, 0, 'h'},
          {0, 0, 0, 0}
        };
      /* getopt_long stores the option index here. */
      int option_index = 0;

      c = getopt_long (argc, argv, "",
                       long_options, &option_index);

      /* Detect the end of the options. */
      if (c == -1)
        break;
      switch (c)
        {
        case 'c':
          compile_only = true;
          break;

        case 'r':
          run_only = true;
          break;

        case 'A':
          break;

        case 'o':
          //printf ("option --tools with value `%s'\n", optarg);
          if (tools_filename == 0) {
            tools_filename = malloc(strlen(optarg)+1);
            strcpy(tools_filename, optarg); 
          } else {
            perror_exit("repeated --tools option\n");
          }
          break;
        case 'e':
          //printf ("option --tests with value `%s'\n", optarg);
          if (tests_filename == 0) {
            tests_filename = malloc(strlen(optarg)+1);
            strcpy(tests_filename, optarg); 
          } else {
            perror_exit("repeated --tests option\n");
          }
          break;
        case 's':
          //printf ("option --test_src_dir with value `%s'\n", optarg);
          if (test_src_dir_filename == 0) {
            test_src_dir_filename = malloc(strlen(optarg)+1);
            strcpy(test_src_dir_filename, optarg); 
          } else {
            perror_exit("repeated --test_src_dir option\n");
          }
            break;
        case 'b':
          //printf ("option --test_bin_dir with value `%s'\n", optarg);
          if (test_bin_dir_filename == 0) {
            test_bin_dir_filename = malloc(strlen(optarg)+1);
            strcpy(test_bin_dir_filename, optarg); 
          } else {
            perror_exit("repeated --test_bin_dir option\n");
          }
            break;
        case 'i':
          //printf ("option --test_int_dir with value `%s'\n", optarg);
          if (test_int_dir_filename == 0) {
            test_int_dir_filename = malloc(strlen(optarg)+1);
            strcpy(test_int_dir_filename, optarg); 
          } else {
            perror_exit("repeated --test_int_dir option\n");
          }
            break;
        case 'l':
          //printf ("option --test_log_dir with value `%s'\n", optarg);
          if (test_log_dir_filename == 0) {
            test_log_dir_filename = malloc(strlen(optarg)+1);
            strcpy(test_log_dir_filename, optarg); 
          } else {
            perror_exit("repeated --test_log_dir option\n");
          }
            break;

        case 'L':
          //printf ("option --log with value `%s'\n", optarg);
          if (log_filename == 0) {
            log_filename = malloc(strlen(optarg)+1);
            strcpy(log_filename, optarg); 
          } else {
            perror_exit("repeated --log option\n");
          }
            break;

        case 'I':
          //printf ("option --int with value `%s'\n", optarg);
          if (int_filename == 0) {
            int_filename = malloc(strlen(optarg)+1);
            strcpy(int_filename, optarg); 
          } else {
            perror_exit("repeated --int option\n");
          }
            break;
            
        case 'h':
          printf ("usage:                                                   \n\
  ./charon-run/charon_run                                                   \n\
     --compile_and_run                  # mode (default)                    \n\
     --tools tools.charon               # the list of tools to use          \n\
     --tests tests.charon               # the list of tests to use          \n\
     --test_src_dir tests/de_facto_memory_model/  # the source dir for tests\n\
     --test_bin_dir tests.bin/          # dir to put compiled binaries      \n\
     --test_log_dir tests.log/          # dir to put per-test log           \n\
     --log tests.log/all.log            # filename to put combined log      \n\
                                                                            \n\
  ./charon-run/charon_run                                                   \n\
     --compile_only                     # mode                              \n\
     --tools tools.charon               # the list of tools to use          \n\
     --tests tests.charon               # the list of tests to use          \n\
     --test_src_dir tests/de_facto_memory_model/  # the source dir for tests\n\
     --test_bin_dir tests.bin/          # dir to put compiled binaries      \n\
     --test_int_dir tests.int/          # dir to put per-test intermediates \n\
     --int tests.int/all.log            # filename to put combined ints     \n\
                                                                            \n\
  ./charon-run/charon_run                                                   \n\
     --run_only                         # mode                              \n\
     --int tests.log/all.log            # filename to get combined ints     \n\
     --test_log_dir tests.log/          # dir to put per-test logs          \n\
     --log tests.log/all.log            # filename to put combined log      \n\
");
          exit(EXIT_SUCCESS); 

        case '?':
          /* getopt_long already printed an error message. */
          break;

        default:
          abort ();
        }
    }


 /* Print any remaining command line arguments (not options). */
  if (optind < argc)
    {
      printf ("error: non-option ARGV-elements: ");
      while (optind < argc)
        printf ("%s ", argv[optind++]);
      putchar ('\n');
      exit(EXIT_FAILURE);
    }

  if (!compile_only && !run_only) {
    mode = COMPILE_AND_RUN;
    if (tools_filename == 0)
      perror_exit("missing --tools argument\n");
    if (tests_filename == 0)
      perror_exit("missing --tests argument\n");
    if (test_src_dir_filename == 0)
      perror_exit("mising --test_src_dir argument\n");
    if (test_bin_dir_filename == 0)
      perror_exit("missing --test_bin_dir argument\n");
    if (test_int_dir_filename != 0) 
      perror_exit("test_int_dir not used in compile-and-run mode\n");
    if (int_filename != 0) 
      perror_exit("int filename not used in compile-and-run mode\n");
    if (test_log_dir_filename == 0)
      perror_exit("missing --test_log_dir argument\n");
    if (log_filename == 0)
      perror_exit("missing --log argument\n");
  }
  else if (compile_only && !run_only) {
    mode = COMPILE_ONLY;
    if (tools_filename == 0)
      perror_exit("missing --tools argument\n");
    if (tests_filename == 0)
      perror_exit("missing --tests argument\n");
    if (test_src_dir_filename == 0)
      perror_exit("mising --test_src_dir argument\n");
    if (test_bin_dir_filename == 0)
      perror_exit("missing --test_bin_dir argument\n");
    if (test_int_dir_filename == 0) 
      perror_exit("missing --test_int_dir argument\n");
    if (int_filename == 0) 
      perror_exit("missing --int argument\n");
    if (test_log_dir_filename != 0) 
      perror_exit("--test_log_dir not used in compile-only mode\n");
    if (log_filename != 0)
      perror_exit("--log filename not used in compile-only mode\n");
  }
  else if (!compile_only && run_only) {
    mode = RUN_ONLY;
    if (tools_filename != 0) 
      perror_exit("--tools not used in run-only mode\n");
    if (tests_filename != 0) 
      perror_exit("--tests not used in run-only mode\n");
    if (test_src_dir_filename != 0) 
      perror_exit("--test_src_dir not used in run-only mode\n");
    if (test_bin_dir_filename != 0) 
      perror_exit("--test_bin_dir not used in run-only mode\n");
    if (test_int_dir_filename != 0) 
      perror_exit("--test_int_dir not used in run-only mode\n");
    if (int_filename == 0)
      perror_exit("missing --int argument\n");
    if (test_log_dir_filename == 0)
      perror_exit("missing --test_log_dir argument\n");
    if (log_filename == 0)
      perror_exit("missing --log argument\n");
    
  }
  else if (compile_only && run_only) {
    perror_exit("cannot have -compile_only and -run_only\n");
  }


  /*
  if (tools_filename == 0) 
    tools_filename = tools_filename_default;
  if (tests_filename == 0) 
    tests_filename = tests_filename_default;
  if (test_src_dir_filename == 0) 
    test_src_dir_filename = test_src_dir_filename_default;
  if (test_bin_dir_filename == 0) 
    test_bin_dir_filename = test_bin_dir_filename_default;
  if (test_int_dir_filename == 0) 
    test_int_dir_filename = test_int_dir_filename_default;
  if (test_log_dir_filename == 0) 
    test_log_dir_filename = test_log_dir_filename_default;
  if (log_filename == 0) 
    log_filename = log_filename_default;
  if (int_filename == 0) 
    int_filename = int_filename_default;
  */



  test_src_dir = string_option_from_constant(test_src_dir_filename);
  test_bin_dir = string_option_from_constant(test_bin_dir_filename);
  test_int_dir = string_option_from_constant(test_int_dir_filename);
  test_log_dir = string_option_from_constant(test_log_dir_filename);


  return;
}

/* ***************************************************************** */
/*  main                                                             */
/* ***************************************************************** */

int main (int argc, char **argv) {

  process_command_line_options(argc, argv);

  host_t h;
  h = get_host();

  if (mode==COMPILE_AND_RUN || mode == COMPILE_ONLY) {

    test_list_t ts = read_tests(tests_filename);
    //pp_tests(stdout, ts);
    tool_list_t tls = read_tools(tools_filename);
    //pp_tools(stdout, tls);
    recipe_list_t rs = make_test_recipes(test_src_dir, ts, tls, NULL);
    int rs_length = length_recipe_list(rs);
    recipe_list_t rs2;
    // printf("recipes:\n");
    // recipe_list_t rs2 = rs;
    // while (rs2!=0) {
    //   pp_recipe(stdout, 0, rs2->recipe,true);
    //   rs2=rs2->next;
    // }
  
    //printf("\n\noutcomes:\n");
    FILE *main_log_or_int_file;
    string_t test_log_or_int_dir;
    if (mode == COMPILE_AND_RUN) {
      test_log_or_int_dir = test_log_dir;
      main_log_or_int_file = fopen(log_filename,"w");
      if (main_log_or_int_file == NULL) {
        fprintf(stderr, "fopen of %s: errno=%d\n",log_filename, errno);
        exit(EXIT_FAILURE);
      }
    }
    else if (mode == COMPILE_ONLY) {
      test_log_or_int_dir = test_int_dir;
      main_log_or_int_file = fopen(int_filename,"w");
      if (main_log_or_int_file == NULL) {
        fprintf(stderr, "fopen of %s: errno=%d\n",int_filename, errno);
        exit(EXIT_FAILURE);
    }
    }
    else
      exit(EXIT_FAILURE);
    fprintf(main_log_or_int_file,"[\n");
  
    rs2=rs;
    for (int count=1; rs2!=0; count++) {
      printf("processing (%i/%i): ",count,rs_length); pp_string(stdout, rs2->recipe->test_instance_name /*rs2->recipe->sources->src_main*/); printf("\n");
      observation_t o = run_test(h,rs2->recipe, test_bin_dir, mode);
      log_observation(o, main_log_or_int_file, test_log_or_int_dir, (count==rs_length));
      rs2=rs2->next;
    }
    fprintf(main_log_or_int_file,"]\n");
    fclose(main_log_or_int_file);

  } else if (mode == RUN_ONLY) {

    int_list_t is; 
    is = read_intermediates(int_filename);
    int is_length = length_int_list(is);


    FILE *main_log_file;
    main_log_file = fopen(log_filename,"w");
    if (main_log_file == NULL) {
      fprintf(stderr, "fopen of %s: errno=%d\n",log_filename, errno);
      exit(EXIT_FAILURE);
    }

    fprintf(main_log_file,"[\n");

    for (int count=1; is != NULL; count++) {
      int_t i;
      i = is->intv;

      printf("processing: (%i/%i)", count, is_length); pp_string(stdout, i->int_binary_filename); printf("\n");


      output_t o;
      o = run_test_from_intermediate(i);
      log_observation_from_intermediate(h, i, o, main_log_file, test_log_dir, (is->next == NULL));
      is = is->next;
    }

    fprintf(main_log_file,"]\n");
    fclose(main_log_file);
  }
  return 0;
}
